/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
*  Copyright (C) 2023--2026, High Performance Kernels LLC                     *
*                                                                             *
*  This software and the related documents are High Performance Kernels LLC   *
*  copyrighted materials, and your use of them is governed by the express     *
*  license under which they were provided to you (License).                   *
*  Unless the License provides otherwise, you may not use, copy, reproduce,   *
*  modify, disclose, transmit, publish, or distribute this software or the    *
*  related documents without prior written permission from High Performance   *
*  Kernels LLC.                                                               *
*                                                                             *
*    This software and the related documents are provided as is, WITHOUT ANY  *
*  WARRANTY, without even the implied warranty of MERCHANTABILITY or FITNESS  *
*  FOR A PARTICULAR PURPOSE.                                                  *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef HPK_ARCHITECTURE_HPP_INCLUDED
#define HPK_ARCHITECTURE_HPP_INCLUDED

/// \file
/// \brief This header defines the Architecture enumeration.

#include <ostream>
#include <string>
#include <type_traits>  // std::underlying_type_t

#include <hpk/visibility.hpp>

namespace hpk {

/// Enumerates values for Parameter::architecture.
enum class Architecture {
    detect = 0,          ///< Architecture will be detected at runtime.
    unsupported = 1,     ///< No supported architecture was detected.
    avx2 = 2,            ///< AVX + AVX2 + FMA on x86_64
    avx512 = 0x12,       ///< Architecture::avx2 + AVX512F + AVX512DQ + AVX512VL
    avx512fp16 = 0x112,  ///< Architecture::avx512 + AVX512_FP16
    sve256 = 0x20100     ///< SVE vector length 256 on AArch64
};

/// \brief True if the left-hand `Architecture` includes the right-hand one.
///
/// For example:
///
///     hpk::Architecture::avx512 >= hpk::Architecture::avx2  // true
///     hpk::Architecture::avx2 >= hpk::Architecture::avx512  // false
///     hpk::Architecture::avx2 >= hpk::Architecture::sve256  // false
///     hpk::Architecture::sve256 >= hpk::Architecture::avx2  // false
///
constexpr bool operator>=(Architecture lhs, Architecture rhs) {
    auto bitsLhs = static_cast<std::underlying_type_t<Architecture>>(lhs);
    auto bitsRhs = static_cast<std::underlying_type_t<Architecture>>(rhs);
    return (bitsLhs & bitsRhs) == bitsRhs;
}

/// \brief True if the left-hand side does not include the right-hand side.
///
/// For example:
///
///     hpk::Architecture::avx512 < hpk::Architecture::avx2  // false
///     hpk::Architecture::avx2 < hpk::Architecture::avx512  // true
///     hpk::Architecture::avx2 < hpk::Architecture::sve256  // true
///     hpk::Architecture::sve256 < hpk::Architecture::avx2  // true
///
constexpr bool operator<(Architecture lhs, Architecture rhs) {
    return !(lhs >= rhs);
}

/// Returns a short string description of an `Architecture`.
HPK_API std::string toString(Architecture arch);

/// Overload for ostream's `<<` operator for an `Architecture`.
inline std::ostream& operator<<(std::ostream& os, Architecture arch) {
    return os << toString(arch);
}

}  // namespace hpk

#endif  // HPK_ARCHITECTURE_HPP_INCLUDED
