/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
*  Copyright (C) 2023--2026, High Performance Kernels LLC                     *
*                                                                             *
*  This software and the related documents are High Performance Kernels LLC   *
*  copyrighted materials, and your use of them is governed by the express     *
*  license under which they were provided to you (License).                   *
*  Unless the License provides otherwise, you may not use, copy, reproduce,   *
*  modify, disclose, transmit, publish, or distribute this software or the    *
*  related documents without prior written permission from High Performance   *
*  Kernels LLC.                                                               *
*                                                                             *
*    This software and the related documents are provided as is, WITHOUT ANY  *
*  WARRANTY, without even the implied warranty of MERCHANTABILITY or FITNESS  *
*  FOR A PARTICULAR PURPOSE.                                                  *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef HPK_COMPLEX_TYPE_TRAITS_HPP_INCLUDED
#define HPK_COMPLEX_TYPE_TRAITS_HPP_INCLUDED

/// \file
/// \brief This header provides for metaprogramming with std::complex types.

#include <complex>
#include <type_traits>

namespace hpk {

/// \brief Used to implement `is_complex_v<T>`.
///
/// If T is complex, provides the member constant `value` equal to true;
/// otherwise, `value` is false.
template<typename T> struct is_complex : std::false_type {};
template<typename T> struct is_complex<std::complex<T>> : std::true_type {};

/// \brief Tests whether a type is `std::complex`.
///
/// For example,
///
///     hpk::is_complex_v<float>                // false
///     hpk::is_complex_v<std::complex<float>>  // true
///
template<typename T> inline constexpr bool is_complex_v = is_complex<T>::value;

/// \brief Used to implement `add_complex_t<T>`.
///
/// Type trait for obtaining a complex type.
/// If T is complex, `type` is T; otherwise, `type` is `std::complex<T>`.
template<typename T> struct add_complex {
    using type = std::complex<T>;
};
template<typename T> struct add_complex<std::complex<T>> {
    using type = std::complex<T>;
};

/// \brief Adds `std::complex` to the given type if it is not already complex.
///
/// For example,
///
///     hpk::add_complex_t<float>                // std::complex<float>
///     hpk::add_complex_t<std::complex<float>>  // std::complex<float>
///
template<typename T> using add_complex_t = typename add_complex<T>::type;

/// \brief Used to implement `remove_complex_t<T>`.
///
/// Type trait for obtaining a real type.
/// If T is complex, `type` is `std::complex::value_type`;
/// otherwise, `type` is T.
template<typename T> struct remove_complex {
    using type = T;
};
template<typename T> struct remove_complex<std::complex<T>> {
    using type = T;
};

/// \brief Removes `std::complex` from the given type.
///
/// For example,
///
///     hpk::remove_complex_t<float>                // float
///     hpk::remove_complex_t<std::complex<float>>  // float
///
template<typename T> using remove_complex_t = typename remove_complex<T>::type;

}  // namespace hpk

#endif  // HPK_COMPLEX_TYPE_TRAITS_HPP_INCLUDED
