/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
*  Copyright (C) 2023--2026, High Performance Kernels LLC                     *
*                                                                             *
*  This software and the related documents are High Performance Kernels LLC   *
*  copyrighted materials, and your use of them is governed by the express     *
*  license under which they were provided to you (License).                   *
*  Unless the License provides otherwise, you may not use, copy, reproduce,   *
*  modify, disclose, transmit, publish, or distribute this software or the    *
*  related documents without prior written permission from High Performance   *
*  Kernels LLC.                                                               *
*                                                                             *
*    This software and the related documents are provided as is, WITHOUT ANY  *
*  WARRANTY, without even the implied warranty of MERCHANTABILITY or FITNESS  *
*  FOR A PARTICULAR PURPOSE.                                                  *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef HPK_CONFIGURATION_HPP_INCLUDED
#define HPK_CONFIGURATION_HPP_INCLUDED

/// \file
/// \brief This header declares Configuration and the Parameter enumeration.

#include <cstddef>  // std::size_t
#include <initializer_list>
#include <ostream>
#include <string>
#include <utility>  // std::move
#include <vector>

#include <hpk/architecture.hpp>
#include <hpk/visibility.hpp>

namespace hpk {

/// Enumerates the parameters that can be set in a `Configuration`.
enum class Parameter {
    invalid = 0x0,       ///< Parameter is invalid
    architecture = 0x1,  ///< value has type `hpk::Architecture`
    threads = 0x100,     ///< value has type int (max total number of threads)
    teams = 0x101,       ///< value has type int (number of teams of threads)
    l1size = 0x111,      ///< value has type int (L1 size measured in KiB)
    l2size = 0x112,      ///< value has type int (L2 size measured in KiB)
    l3size = 0x113,      ///< value has type int (L3 size measured in KiB)
};

/// \brief A single configuration setting.
///
/// This class holds a Parameter and its associated value.
///
class HPK_API ConfigItem {
 private:
    Parameter param;
    int value;

    friend class Configuration;

 public:
    /// Default constructor.
    constexpr ConfigItem() : param(Parameter::invalid), value(0) {}

    /// Constructs a `ConfigItem` for the specified machine architecture.
    constexpr ConfigItem(Architecture arch)  // NOLINT
        : param(Parameter::architecture), value(static_cast<int>(arch)) {}

    // Constructs a `ConfigItem` for the specified machine architecture.
    constexpr ConfigItem(Parameter, Architecture arch) : ConfigItem(arch) {}

    /// Constructs a `ConfigItem` for a parameter `p` whose value is an int.
    constexpr ConfigItem(Parameter p, int v) : param(p), value(v) {}

    // Computes a hash based only on the param, ignoring the value.
    std::size_t operator()(const ConfigItem& item) const noexcept;
    // Two objects are equal if and only if they have the same param.
    friend bool operator==(const ConfigItem& lhs, const ConfigItem& rhs);
    friend bool operator!=(const ConfigItem& lhs, const ConfigItem& rhs);

    std::string toString() const;
    friend std::ostream& operator<<(std::ostream& os, const ConfigItem& item);
};

/// A configuration setting that specifies single-threaded computation.
inline constexpr ConfigItem sequential{Parameter::threads, 1};

/// \brief A collection of configuration settings.
///
/// A `Configuration` is a container for parameters and their associated values.
///
/// Example:
///
///     hpk::Configuration cfg{hpk::Architecture::avx2, hpk::sequential};
///
/// or, equivalently,
///
///     hpk::Configuration cfg{
///            {hpk::Parameter::architecture, hpk::Architecture::avx2},
///            {hpk::Parameter::threads,      1}
///     };
///
class HPK_API Configuration {
 private:
    std::vector<ConfigItem> config;

 public:
    /// Default constructor.
    ///
    Configuration() = default;

    /// Constructs a `Configuration` instance from an initializer list.
    /// Each `ConfigItem` in the list is either an `Architecture` or a
    /// key and integer value, `{Parameter, int}`.
    /// If multiple entries in the list have the same parameter, it is
    /// unspecified which one will be used.
    Configuration(std::initializer_list<ConfigItem> init) : config(init) {}

    /// Constructs a `Configuration` instance from a vector by copying
    /// its contents.
    Configuration(const std::vector<ConfigItem>& v) : config(v) {}  // NOLINT

    /// Constructs a `Configuration` instance from a vector using move
    /// semantics.
    Configuration(std::vector<ConfigItem>&& v) noexcept  // NOLINT
        : config(std::move(v)) {}

    /// Returns the `Architecture` that was set in this configuration.
    /// If an architecture was not set, it returns `Architecture::detect`.
    Architecture getArchitecture() const;

    /// Returns the value of the specified parameter that was set in this
    /// configuration.
    /// If a value for the parameter was not set, it returns 0.
    int get(Parameter p) const;

    /// Returns a string description of a `Configuration`.
    std::string toString() const;

    /// Overload for ostream's `<<` operator for a `Configuration`.
    friend std::ostream& operator<<(std::ostream& os, const Configuration& cfg);
};

}  // namespace hpk

#endif  // HPK_CONFIGURATION_HPP_INCLUDED
