/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
*  Copyright (C) 2023--2026, High Performance Kernels LLC                     *
*                                                                             *
*  This software and the related documents are High Performance Kernels LLC   *
*  copyrighted materials, and your use of them is governed by the express     *
*  license under which they were provided to you (License).                   *
*  Unless the License provides otherwise, you may not use, copy, reproduce,   *
*  modify, disclose, transmit, publish, or distribute this software or the    *
*  related documents without prior written permission from High Performance   *
*  Kernels LLC.                                                               *
*                                                                             *
*    This software and the related documents are provided as is, WITHOUT ANY  *
*  WARRANTY, without even the implied warranty of MERCHANTABILITY or FITNESS  *
*  FOR A PARTICULAR PURPOSE.                                                  *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef HPK_DETECTION_HPP_INCLUDED
#define HPK_DETECTION_HPP_INCLUDED

/// \file
/// \brief This header declares functions for detecting system properties.

#include <string>

#include <hpk/architecture.hpp>
#include <hpk/visibility.hpp>

/// The version of this release of the Hpk library, as a C string.
extern "C" HPK_API const char hpk_version[];

namespace hpk {

/// Provides processor cache information.
struct CacheInfo {
    unsigned int size;      ///< Total size of the cache in KiB
    int sharedCpus;         ///< Number of CPUs sharing the cache
    unsigned int lineSize;  ///< Size of each cache line in bytes
    unsigned int sets;      ///< Number of sets (cache lines per way)
    unsigned int ways;      ///< Associativity (cache lines per set)
};

/// Returns a string description of a `CacheInfo`.
/// \related CacheInfo
HPK_API std::string toString(const CacheInfo& info);

/// Overload for ostream's `<<` operator for a `CacheInfo`.
/// \related CacheInfo
inline std::ostream& operator<<(std::ostream& os, const CacheInfo& info) {
    return os << toString(info);
}

/// Returns the instruction set architecture available to the current process.
HPK_API Architecture detectArchitecture();

/// \brief Returns the number of CPUs (i.e., hardware threads) available to the
///        current process.
HPK_API int detectCpus();

/// Returns the number of cores available to the current process.
HPK_API int detectCores();

/// Returns the number of dies available to the current process.
HPK_API int detectDies();

/// Returns the number of packages available to the current process.
HPK_API int detectPackages();

/// \brief Returns information describing the hardware data cache at the
///        specified level for the CPU running the current process.
HPK_API CacheInfo detectDataCache(int cacheLevel);

}  // namespace hpk

#endif  // HPK_DETECTION_HPP_INCLUDED
